using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Threading;

namespace SPStudio
{
	/// <summary>
	/// An implementation of
	/// the Volumetric Reverse 
	/// Multiple Correspondence Algorithm.
	/// </summary>
	public class VRMCAlg : System.Windows.Forms.Form
	{
		#region Public Vars
		public Workspace workspace = null;
		public ArrayList region1Triangles = null;
		public ArrayList region2Triangles = null;
		public ArrayList region3Triangles = null;
		public ArrayList region4Triangles = null;
		public ArrayList shiftVoxels;
		public Vector3D [,,] vectors;
		public int vectorRows = 0;
		public int vectorCols = 0;
		#endregion

		#region Private Vars
		private Thread runAlg;
		#endregion

		#region Windows Generated Vars
		private System.Windows.Forms.ProgressBar ProgressBar;
		private System.Windows.Forms.Label StateLabel;
		private System.Windows.Forms.Panel ButtonPanel;
		private System.Windows.Forms.Button OkayButton;
		private System.Windows.Forms.Button Cancel_Button;
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		#endregion

		#region Constructor
		/// <summary>
		/// Obtain a handles to required
		/// parent workspace variables.
		/// Gather the Triangle3Ds grouped
		/// by region.
		/// </summary>
		/// <param name="Parent"></param>
		/// <param name="Region1Triangles"></param>
		/// <param name="Region2Triangles"></param>
		/// <param name="Region3Triangles"></param>
		/// <param name="Region4Triangles"></param>
		public VRMCAlg(Workspace Parent, ArrayList Region1Triangles, ArrayList Region2Triangles,
			ArrayList Region3Triangles, ArrayList Region4Triangles)
		{
			workspace = Parent;
			region1Triangles = Region1Triangles;
			region2Triangles = Region2Triangles;
			region3Triangles = Region3Triangles;
			region4Triangles = Region4Triangles;
			shiftVoxels = Parent.shiftVoxels;
			vectors = Parent.vectors;
		
			InitializeComponent();
		}
		#endregion

		#region Dispose Operation
		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}
		#endregion

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.ProgressBar = new System.Windows.Forms.ProgressBar();
			this.StateLabel = new System.Windows.Forms.Label();
			this.ButtonPanel = new System.Windows.Forms.Panel();
			this.Cancel_Button = new System.Windows.Forms.Button();
			this.OkayButton = new System.Windows.Forms.Button();
			this.ButtonPanel.SuspendLayout();
			this.SuspendLayout();
			// 
			// ProgressBar
			// 
			this.ProgressBar.Dock = System.Windows.Forms.DockStyle.Top;
			this.ProgressBar.Enabled = false;
			this.ProgressBar.Location = new System.Drawing.Point(0, 32);
			this.ProgressBar.Name = "ProgressBar";
			this.ProgressBar.Size = new System.Drawing.Size(456, 24);
			this.ProgressBar.TabIndex = 1;
			// 
			// StateLabel
			// 
			this.StateLabel.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.StateLabel.Dock = System.Windows.Forms.DockStyle.Top;
			this.StateLabel.Enabled = false;
			this.StateLabel.Location = new System.Drawing.Point(0, 0);
			this.StateLabel.Name = "StateLabel";
			this.StateLabel.Size = new System.Drawing.Size(456, 32);
			this.StateLabel.TabIndex = 2;
			this.StateLabel.Text = "Waiting...";
			this.StateLabel.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			// 
			// ButtonPanel
			// 
			this.ButtonPanel.Controls.Add(this.Cancel_Button);
			this.ButtonPanel.Controls.Add(this.OkayButton);
			this.ButtonPanel.Dock = System.Windows.Forms.DockStyle.Top;
			this.ButtonPanel.Location = new System.Drawing.Point(0, 56);
			this.ButtonPanel.Name = "ButtonPanel";
			this.ButtonPanel.Size = new System.Drawing.Size(456, 32);
			this.ButtonPanel.TabIndex = 3;
			// 
			// Cancel_Button
			// 
			this.Cancel_Button.Dock = System.Windows.Forms.DockStyle.Right;
			this.Cancel_Button.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.Cancel_Button.Location = new System.Drawing.Point(232, 0);
			this.Cancel_Button.Name = "Cancel_Button";
			this.Cancel_Button.Size = new System.Drawing.Size(224, 32);
			this.Cancel_Button.TabIndex = 1;
			this.Cancel_Button.Text = "Cancel";
			this.Cancel_Button.Click += new System.EventHandler(this.Cancel_Button_Click);
			// 
			// OkayButton
			// 
			this.OkayButton.Dock = System.Windows.Forms.DockStyle.Left;
			this.OkayButton.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.OkayButton.Location = new System.Drawing.Point(0, 0);
			this.OkayButton.Name = "OkayButton";
			this.OkayButton.Size = new System.Drawing.Size(232, 32);
			this.OkayButton.TabIndex = 0;
			this.OkayButton.Text = "OK";
			this.OkayButton.Click += new System.EventHandler(this.OkayButton_Click);
			// 
			// VRMCAlg
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(456, 86);
			this.ControlBox = false;
			this.Controls.Add(this.ButtonPanel);
			this.Controls.Add(this.ProgressBar);
			this.Controls.Add(this.StateLabel);
			this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.Fixed3D;
			this.Name = "VRMCAlg";
			this.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
			this.Text = "VRMC Algorithm";
			this.ButtonPanel.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Threaded Method
		/// <summary>
		/// Threaded method that is the bulk
		/// of the algorithm.
		/// </summary>
		private void makeShiftVoxels()
		{
			double xMax = workspace.greatestX;
			double yMax = workspace.greatestY;
			double zMax = workspace.greatestX;

			double xMin = 0.0-workspace.greatestX;
			double yMin = workspace.smallestY;
			double zMin = 0.0-workspace.greatestX;

			double voxelSize = 0.0;

			//Generate voxel size by using a ratio,
			//this way voxel detail is not determined
			//by the resolution of the images.
			if(workspace.getSize().Width>workspace.getSize().Height)
			{
				voxelSize = (double)workspace.getSize().Width/90.0;
			}
			else
			{
				voxelSize = (double)workspace.getSize().Height/90.0;
			}

			//Create a one voxel cushion
			xMax+=voxelSize;
			xMin-=voxelSize;
			yMax+=voxelSize;
			yMin-=voxelSize;
			zMax+=voxelSize;
			zMin-=voxelSize;
		
			//Determine the rows and columns of
			//voxels needed to fill the 3D space
			//occupied by the target object.
			int columns = (int)Math.Ceiling((Math.Abs(xMax) + Math.Abs(xMin))/voxelSize);
			int rows = (int)Math.Ceiling((Math.Abs(yMax) + Math.Abs(yMin))/voxelSize);

			//create vector array
			vectorRows = rows;
			vectorCols = columns;
			vectors = new Vector3D[columns,rows,columns];

			double xLocation;
			double yLocation;
			double zLocation = zMin;

			StateLabel.Text = "Creating Vectors";
			ProgressBar.Step = 1;
			ProgressBar.Value = 0;
			ProgressBar.Minimum = 0;
			ProgressBar.Maximum = (columns*rows*columns);
			Show();
			Refresh();
			//Loop to create the vectors.
			for(int zIndex=0; zIndex<columns; zIndex++)
			{
				yLocation = yMin;
				for(int yIndex=0; yIndex<rows; yIndex++)
				{
					xLocation = xMin;
					for(int xIndex=0; xIndex<columns; xIndex++)
					{
						vectors[xIndex,yIndex,zIndex] = new Vector3D(xLocation,yLocation,zLocation);
						ProgressBar.PerformStep();
						xLocation+=voxelSize;
					}
					yLocation+=voxelSize;
				}
				zLocation+=voxelSize;
			}
			ProgressBar.Value = 0;

			//Associate the vectors with voxels

			//This is because the number of voxels in a column/row
			//is alwasy n-1 where n is the number of vectors
			//in the column/row
			columns--;
			rows--;

			ArrayList voxels = new ArrayList(1);

			StateLabel.Text = "Creating Voxels";
			ProgressBar.Maximum = (columns*rows*columns);
			Refresh();
			//Loop to associate vectors with voxels.
			for(int zIndex=0; zIndex<columns; zIndex++)
			{
				for(int yIndex=0; yIndex<rows; yIndex++)
				{
					for(int xIndex=0; xIndex<columns; xIndex++)
					{
						voxels.Add(new Voxel(
							vectors[xIndex,yIndex,zIndex],
							vectors[xIndex+1,yIndex,zIndex],
							vectors[xIndex+1,yIndex+1,zIndex],
							vectors[xIndex,yIndex+1,zIndex],
							vectors[xIndex,yIndex,zIndex+1],
							vectors[xIndex+1,yIndex,zIndex+1],
							vectors[xIndex+1,yIndex+1,zIndex+1],
							vectors[xIndex,yIndex+1,zIndex+1]));
						voxels.Capacity++;
						ProgressBar.PerformStep();
					}
				}
			}
			ProgressBar.Value = 0;

			StateLabel.Text = "Calculating Intersections";
			ProgressBar.Maximum = ((columns+1)*(rows+1)*(columns+1));
			IEnumerator vectorWalk = vectors.GetEnumerator();

			Refresh();
			//Intersection testing.
			while(vectorWalk.MoveNext())
			{
				Vector3D v = (Vector3D)vectorWalk.Current;

				//This is a vector know to be outside of the target object
				Vector3D baseVector = new Vector3D(0.0,yMax*2.0,0.0);

				Line3D testLine = new Line3D(v,baseVector);
				IEnumerator triangleGroupWalk = region1Triangles.GetEnumerator();
				//Test Region1
				while(triangleGroupWalk.MoveNext()&&v.in0to90)
				{
					int intersections = 0;

					ArrayList triangles = (ArrayList)triangleGroupWalk.Current;
					IEnumerator triangleWalk = triangles.GetEnumerator();
					while(triangleWalk.MoveNext())
					{
						Triangle3D t = (Triangle3D)triangleWalk.Current;
						if(testLine.intersectsTriangle(t))
						{
							intersections++;
						}
					}

					if((intersections==0)||(intersections%2==0))
					{							
						v.in0to90 = false;
					}
				}

				triangleGroupWalk = region2Triangles.GetEnumerator();
				//Test Region2
				while(triangleGroupWalk.MoveNext()&&v.in90to180)
				{
					int intersections = 0;

					ArrayList triangles = (ArrayList)triangleGroupWalk.Current;
					IEnumerator triangleWalk = triangles.GetEnumerator();
					while(triangleWalk.MoveNext())
					{
						Triangle3D t = (Triangle3D)triangleWalk.Current;
						if(testLine.intersectsTriangle(t))
						{
							intersections++;
						}
					}

					if((intersections==0)||(intersections%2==0))
					{							
						v.in90to180 = false;
					}
				}

				triangleGroupWalk = region3Triangles.GetEnumerator();
				//Test Region3
				while(triangleGroupWalk.MoveNext()&&v.in180to270)
				{
					int intersections = 0;

					ArrayList triangles = (ArrayList)triangleGroupWalk.Current;
					IEnumerator triangleWalk = triangles.GetEnumerator();
					while(triangleWalk.MoveNext())
					{
						Triangle3D t = (Triangle3D)triangleWalk.Current;
						if(testLine.intersectsTriangle(t))
						{
							intersections++;
						}
					}

					if((intersections==0)||(intersections%2==0))
					{							
						v.in180to270 = false;
					}
				}

				triangleGroupWalk = region4Triangles.GetEnumerator();
				//Test Region4
				while(triangleGroupWalk.MoveNext()&&v.in270to360)
				{
					int intersections = 0;

					ArrayList triangles = (ArrayList)triangleGroupWalk.Current;
					IEnumerator triangleWalk = triangles.GetEnumerator();
					while(triangleWalk.MoveNext())
					{
						Triangle3D t = (Triangle3D)triangleWalk.Current;
						if(testLine.intersectsTriangle(t))
						{
							intersections++;
						}
					}

					if((intersections==0)||(intersections%2==0))
					{							
						v.in270to360 = false;
					}
				}
				ProgressBar.PerformStep();
				Thread.Sleep(0);
			}
			ProgressBar.Value = 0;
			shiftVoxels = voxels;
			Close();
		}
		#endregion

		#region Methods to Retreive Data
		public ArrayList getShiftVoxels()
		{
			return shiftVoxels;
		}

		public Vector3D [,,] getVectors()
		{
			return vectors;
		}

		public int getVectorRows()
		{
			return vectorRows;
		}

		public int getVectorCols()
		{
			return vectorCols;
		}
		#endregion

		#region Button Handlers
		/// <summary>
		/// Trigger the algorithm start.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OkayButton_Click(object sender, System.EventArgs e)
		{
			OkayButton.Enabled=false;
			runAlg = new Thread(new ThreadStart(makeShiftVoxels));
			runAlg.Start();
		}

		/// <summary>
		/// Cancel the algorithm run.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Cancel_Button_Click(object sender, System.EventArgs e)
		{
			try
			{
				runAlg.Abort();
			}
			catch(NullReferenceException)
			{
			}
			Close();
		}
		#endregion
	}
}
